"""Service to configure APScheduler jobs for sending reminders."""

from __future__ import annotations

from apscheduler.schedulers.background import BackgroundScheduler
from apscheduler.triggers.cron import CronTrigger
from zoneinfo import ZoneInfo
from datetime import datetime, timedelta

from ca_task_manager.services.email_service import EmailService
from ca_task_manager.services.settings_service import SettingsService
from ca_task_manager.utils.misc import now_in_timezone


class SchedulerService:
    def __init__(self, email_service: EmailService = None, settings_service: SettingsService = None):
        self.email_service = email_service or EmailService()
        self.settings_service = settings_service or SettingsService()
        self.scheduler = None

    def send_due_reminders(self) -> None:
        """Wrapper around EmailService.send_due_reminders for APScheduler."""
        # Use reference date in local timezone
        tz_name = self.settings_service.get_settings().timezone
        reference_date = now_in_timezone(tz_name).date()
        self.email_service.send_due_reminders(reference_date)

    def start_scheduler(self) -> None:
        """Start the APScheduler to run the reminder job daily at 9:00 AM local time."""
        settings = self.settings_service.get_settings()
        tz = ZoneInfo(settings.timezone)
        scheduler = BackgroundScheduler(timezone=tz)
        # 9:00 AM trigger
        trigger = CronTrigger(hour=9, minute=0)
        scheduler.add_job(self.send_due_reminders, trigger)
        scheduler.start()
        self.scheduler = scheduler
        try:
            # Keep scheduler running
            import time

            while True:
                time.sleep(60)
        except (KeyboardInterrupt, SystemExit):
            scheduler.shutdown()